package com.ejie.ab04b.util;

import java.io.IOException;
import java.util.List;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.context.support.ReloadableResourceBundleMessageSource;

import com.ejie.ab04b.constantes.Constantes;
import com.ejie.ab04b.exception.ErrorField;
import com.ejie.ab04b.exception.Mensaje;

/**
 * ErroresViewHelper.
 * 
 * @author GFI
 */
public final class ErroresViewHelper {

	// Literal de titulo del error
	private static final String ERROR_TITLE = "error.titulo";

	/**
	 * Constructor de ErroresViewHelper.
	 */
	private ErroresViewHelper() {
	}

	/**
	 * Metodo que convierte una lista de ErrorField en la sálida a Formato html
	 * para su tratamiento por el FeedBack de Errores. Formato p.ej.:
	 * 
	 * //
	 * <ul class="rup-maint_feedbackUL">
	 * Se han producido los siguientes errores:<br>
	 * <li><b>DNI:</b>
	 * <ul>
	 * <li>El campo es obligatorio.</li>
	 * </ul>
	 * </li>
	 * </ul>
	 * 
	 *  messageSource
	 *            mensajes
	 *  errorFields
	 *            lista de errorField
	 *  String
	 *
	 * @param messageSource the message source
	 * @param errorFields the error fields
	 * @return the string
	 */
	// CHECKSTYLE:OFF CyclomaticComplexity
	public static String obtenerErrorRup(
			ReloadableResourceBundleMessageSource messageSource,
			List<ErrorField> errorFields) {
		StringBuilder stb = new StringBuilder();
		if (!com.gfi.utils.Utilities.isEmptyList(errorFields)) {
			// Generar Inicio
			stb.append("<ul class=\"rup-maint_feedbackUL\">");
			stb.append(Utilities.getLiteralMessage(messageSource,
					ErroresViewHelper.ERROR_TITLE));
			stb.append("<br/>");

			boolean hayCampo = false;
			for (ErrorField errorField : errorFields) {
				hayCampo = false;
				if (errorField.getCampo() != null) {
					hayCampo = true;
					// Generar Campo
					stb.append("<li>").append("<b>");
					stb.append(Utilities.getLiteralMessage(messageSource,
							errorField.getCampo()));
					stb.append(":</b>");
				}
				boolean hayMensaje = false;
				for (Mensaje mensaje : errorField.getMensajes()) {
					// Generar Mensaje
					if (hayCampo && !hayMensaje) {
						stb.append("<ul>");
					}
					if (!hayCampo) {
						stb.append("<li>").append("<b>");
					} else {
						stb.append("<li>");
					}
					stb.append(mensaje.isTraducido() ? mensaje.getKey()
							: Utilities
									.getInstance()
									.getLiteralMessageWithParams(
											messageSource,
											mensaje.getKey(),
											Utilities
													.getInstance()
													.parseParametroKeyToObject(
															messageSource,
															mensaje.getParametrosKey())));
					if (!hayCampo) {
						stb.append("</b>");
						stb.append("</li>");
					} else {
						stb.append("</li>");
					}
					hayMensaje = true;
				}
				if (hayCampo && hayMensaje) {
					stb.append("</ul>");
				}
				if (hayCampo) {
					stb.append("</li>");
				}
			}
			stb.append("</ul>");
		}
		return stb.toString();
	}

	// CHECKSTYLE:ON CyclomaticComplexity

	/**
	 * Metodo que construye la respuesta para llamadas desde IE para escribir en
	 * el textarea el resultado.
	 * 
	 *  request
	 *            HttpServletRequest
	 *  response
	 *            HttpServletResponse
	 *  introduceTextArea
	 *            boolean Indica si es necesario escribir el textarea. Caso
	 *            especial de subida máxima permitida superada.
	 *  texto
	 *            String
	 *
	 * @param request the request
	 * @param response the response
	 * @param introduceTextArea the introduce text area
	 * @param texto the texto
	 * @throws IOException             e
	 */
	public static void writeRdoIframe(HttpServletRequest request,
			HttpServletResponse response, boolean introduceTextArea,
			String texto) throws IOException {
		ServletOutputStream outputStream = response.getOutputStream();
		if (introduceTextArea) {
			response.setStatus(HttpServletResponse.SC_OK);
		} else {
			response.setStatus(HttpServletResponse.SC_NOT_ACCEPTABLE);
		}
		response.setContentType("text/plain");
		response.setCharacterEncoding(Constantes.ENCODING_UTF8);
		if (introduceTextArea) {
			outputStream.write("<textarea status='406' statusText=''>"
					.getBytes(Constantes.ENCODING_UTF8));
		}
		outputStream.write(texto.getBytes(Constantes.ENCODING_UTF8));
		if (introduceTextArea) {
			outputStream
					.write("</textarea>".getBytes(Constantes.ENCODING_UTF8));
		}
		outputStream.flush();
		outputStream.close();

		response.flushBuffer();
	}
}